function [Got]=STM32F411_ADC_REC(skt,FileName,DS_Ratio,n)
%
%function [Got]=STM32F411_ADC_REC(skt,FileName,DS_Ratio,n)
%
% Get Data and optionally plot from an STM32F411 board with peek poke
%  skt           : peek poke udp skt structure
%  FileName      : name of file to record data in
%  DS_Ratio      : Downsample ratio - integer - if >1 then lopass filter and downsample
%                : Filter edge is at 0.75 times nyquist - i.e 0.3725 output sample rate
%                : If DS_Ratio >1 then will time align channels by
%                : fractionally shifting the decimation lowpass filter according to the channel delay data in Meta.Chandelays
%                : if DS_Ratio is absent assumes 1
%  n             : number of sample frames to get
%                : if n is absent - then record until space pressed
%
%                :
%  Got           : Number of sample frames saved in file
%
% See Also STM32F411_ADC_SETUP.m STM32F411_ADC_PLOT.m ACQUIRE_FILE_TAIL_PLOT.m
%           ACQUIRE_FILE_GET
%
%   Ian Stothers Mar-Apr 2021
% Apr Changed default ordering in input params
%
%   To Do
% Some indication of data loss because we can't keep up
% This is more likely as the sample rate * no of channels gets above 80k values/sec
% - (i.e data rate above 160kbytes/sec)
% perhaps chane n to be in terms of output samples
% Add time align - done but efficiency could be improved - downsamples on output of fir1
% Add option for headless record.
% Add filter corner parameter
% Display IP address of STM32
%
%
forever=0;
  pkg unload signal
    pkg load signal
if (nargin<4)
  forever=1;
  n=1e12;
endif

if (nargin<3)
  DS_Ratio=1;
endif
flen=DS_Ratio*31;
if (DS_Ratio>1)
 n=((n+1)*DS_Ratio)+(flen);
end

ADC1_CAL=udpget(skt,"ADC1_CAL");
STM32F411_bitfields; % Get some bitfields

bpe=[1 2 4 4](1+bitand(floor(udpget(skt,"DMA2_S0_CR")/(2^13)),3));  % Bytes per element from the DMA engine
SQR1=udpget(skt,"ADC1_SQR1");
nchan = bitand(floor(SQR1/2^20),15)+1;           % Number of Channels from the ADC engine
DMABUF_ADDR=skt.ptr(skt_find_index(skt,"ADC_DMA_Buffer"));          % DMA buffer address from the registry
msdma=floor(skt.len(skt_find_index(skt,"ADC_DMA_Buffer"))/bpe);            % DMA buffer size in bpe lumps from the registry
FT_Clock=double(udpget(skt,"SystemCoreClock"));                             % Read Core Clock rate from registry
T2_PSC=double(udpget(skt,"TIM2_PSC"));                                      % Read T2 presclaer from registry
T2_ARR=double(udpget(skt,"TIM2_ARR"));                                      % Read T2 Period (ARR) Reg from registry

DT=((T2_PSC+1.0)*(T2_ARR+1.0))/FT_Clock;
SAMP_CONST=bitand(udpget(skt,"ADC1_SMPR1"),7);
ADC_PR=bitand(ADC_PRE,udpget(skt,"ADC1_COMMON_CCR"))/ADC_PRE;
CR1=udpget(skt,"ADC1_CR1");
RES_VAL=bitand(CR1,RES*3)/RES;
G_Val =1.0; % G_Val is a frig - on left align for whatever reason the chip puts the adc result in the LSB
if (RES_VAL==3)
 G_Val =256.0;
end
SampCycles=[3 15 28 56 84 112 144 480]; % lookup for SampCycles(SAMP_CONST+1)
ChanDelays=((1:nchan)-1)*((3+SampCycles(SAMP_CONST+1)+(15-2*RES_VAL))*(2+(2*(ADC_PR))))/FT_Clock;

s_of_dma=floor(msdma/nchan)*nchan;                                  % ensure dma is shrunk to contain integer number of frames to keep channel alignment
n=n*nchan; % total number of samples to get

if (exist("DS_Ratio")>0)
 if (DS_Ratio>1)
  if (exist("fir1")==0)
   pkg load signal;
  end
  NiqFrac=0.8;
  b =fir1(flen-1,NiqFrac/DS_Ratio);
  [bcoef]=dfir(b , (ChanDelays/DT))';
  bcoef=flipud(bcoef); % time revrse filters - and delays
  ChanDelays=ChanDelays*0;
  YS=zeros(flen,nchan);
 end
end
Meta.ChanDelays=ChanDelays;
Meta.NumChan=nchan;
Meta.Clock=clock;
Meta.DeltaT=DT*DS_Ratio;
Meta.StartupTransientSamples=ceil(flen/DS_Ratio);

save(FileName,"Meta");
fh=fopen(FileName,"a");
fwrite(fh,0,"char");
fclose(fh);
 % Status and stop figure
 sf=figure ("toolbar", "none");
  set(sf,'numbertitle','off');
 set(sf,'resize','on');
 set(sf,'renderermode','manual');
 set(sf,'menubar','none');
 set(sf,'name','STM32F411_ADC_REC');
% set(sf,'position',[10 200 300 150]);  % must be last window configuring item - don;t know why
  set(sf,'position',[10 250 350 400]);  % must be last window configuring item - don;t know why
 [sb]=stopbut(sf,[75 5 195 22],'END ACQUIRE');
 St='';
 th_cnt=text(0.05,0.9,St);
 set(th_cnt,'fontsize',12);
 th_chan=text(0.05,0.8,sprintf('Number of Channels - %d',nchan));
 set(th_chan,'fontsize',12);
 th_sr=text(0.05,0.7,sprintf('Record Sample Rate - %f',1/Meta.DeltaT));
 set(th_sr,'fontsize',12);
 th_sra=text(0.05,0.6,sprintf('ADC Sample Rate - %f',1/DT));
 set(th_sra,'fontsize',12);
 th_fn=text(0.05,0.5,['Acquire Filename - ' FileName]);
 set(th_fn,'fontsize',12);
 if (DS_Ratio>1)
  F3dB=NiqFrac*0.5/(DS_Ratio*DT);
  tFs=sprintf('Filter 3dB - %f Hz - Time Aligned',F3dB);
 else
  tFs='No Filter - No Channel Time Alignment';
 endif
 th_fi=text(0.05,0.4,tFs);
 set(th_fi,'fontsize',12);
 th_ip=text(0.05,0.3, sprintf('IP - %s', skt.ip));
 set(th_ip,'fontsize',12);

 axis('off')

 Data=[];
 DataOut=[];
 N=0;
 Got=0;
 kb=0;

 tail=(s_of_dma-(ceil(udpget(skt,'DMA2_S0_NDTR')/nchan)*nchan));
 tt=tic;
 oldtoc=toc(tt);
 fh=fopen(FileName,'a');
 while ((N<n)||(forever==1))
  St= sprintf( 'Acquire Count - %d',floor(Got/(nchan*DS_Ratio)));
  set(th_cnt,'string',St);
  kb=get(sb,'value');
  if(kb>0)
   forever=0;
   N=n;
  endif
  ptr=(s_of_dma-(ceil(udpget(skt,'DMA2_S0_NDTR')/nchan)*nchan));
  if (tail>ptr)
   toget=min(s_of_dma-tail,n-N);
   data=udppeek(skt,DMABUF_ADDR+(tail*bpe),toget*bpe);
   Data=[Data data];
   N=N+toget;
   Got=Got+toget;
   tail=tail+toget;
   if (tail>=s_of_dma)
     tail=0;
   endif
  endif
  if (((ptr>tail)&&(N<n))||(forever==1))
   toget=min(ptr-tail,n-N);
   data=udppeek(skt,DMABUF_ADDR+(tail*bpe),toget*bpe);
   Data=[Data data];
   N=N+toget;
   Got=Got+toget;
   tail=tail+toget;
  endif
  if(min(size(Data))>0)
   ttmp=toc(tt);
   tc=ttmp-oldtoc;
   oldtoc=ttmp;

   if (tc<0.1)
    pause(max([0.1-tc 0.01]));
   end
    Data=udpgettconv(Data,[0 2](bpe),nchan); % 0 is uint8  2 is uint16
    Data=double(Data*ADC1_CAL*G_Val);
    if (DS_Ratio>1)
     Data=Data';
     Dataf=zeros(size(Data));
% separate filters for each channel to prepare for time alignment
     for c=1:nchan
      [Dataf(:,c),YS(:,c)]=filter(bcoef(:,c),1,Data(:,c),YS(:,c));
     endfor
     DataOut=[DataOut Dataf'];
     md=size(DataOut)(2);
     if(md>(2*DS_Ratio))
      sdtosave=(floor((md/DS_Ratio))*DS_Ratio);
      svec=1:DS_Ratio:(sdtosave);
      fwrite(fh,DataOut(:,svec(1:(end-1)))(:),"single");
      if ((md-(svec(end-1)))>0)
       DataOut=DataOut(:,svec(end):md);
      else
       DataOut=[];
      end %((md-(svec(end-1)))>0)
     end  %if(md>(2*DS_Ratio))
    else
     fwrite(fh,Data,'single');
   end
   Data=[];
  end
  drawnow;
  fflush(fh);
 endwhile
 fclose(fh);
 Got=Got/nchan;
 close(sf);
endfunction

function [h]=dfir(b,delay)
%function [h]=dfir(b,delay)
% construct a set of filters based on an input filter impulse response
% with delays of up to 1 sample
% delay is in fractions of a sample period
% Ian Stothers Apr 2021
 msd=max(size(delay));
 msb=max(size(b));
 bo=b;
 b=[0 b];
 h=zeros(msd,msb+1);
 usr=32;
 bus=resample(b,usr,1);
 msbus=max(size(bus));
 dtb=1/usr;
 tvecu=(0:(msbus-1))*dtb;
 tveco=0:msb;
 for f=1:msd
  tvecd=tveco+delay(f);
  h(f,:)=interp1(tvecu,bus,tvecd);
 end
end


function [pb]=stopbut(f,pos,txt)
 pb=uicontrol(...
 'style','togglebutton',...
 'string',txt,...
 'position',pos);
end



